"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsLoadBalancerController = void 0;
const eks = require("@aws-cdk/aws-eks");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
// eslint-disable-next-line
const request = require('sync-request');
// eslint-disable-next-line
const yaml = require('js-yaml');
class AwsLoadBalancerController extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Adopted from comments in this issue: https://github.com/aws/aws-cdk/issues/8836
        const albServiceAccount = props.cluster.addServiceAccount('aws-alb-ingress-controller-sa', {
            name: 'aws-load-balancer-controller',
            namespace: 'kube-system',
        });
        const awsAlbControllerPolicyUrl = 'https://raw.githubusercontent.com/kubernetes-sigs/aws-load-balancer-controller/v2.2.0/docs/install/iam_policy.json';
        const policyJson = request('GET', awsAlbControllerPolicyUrl).getBody('utf8');
        (JSON.parse(policyJson)).Statement.forEach(statement => {
            albServiceAccount.addToPrincipalPolicy(iam.PolicyStatement.fromJson(statement));
        });
        // AWS Load Balancer Controller Helm Chart
        // https://kubernetes-sigs.github.io/aws-load-balancer-controller/v2.2/deploy/installation/#summary
        // AWS Application Load Balancer Controller Helm Chart CRDs
        // These must be installed before installing the Helm Chart because CDK
        // installs the Helm chart with `helm upgrade` which does not
        // automatically install CRDs
        // This should be equivalent to Step 2:
        // kubectl apply -k "github.com/aws/eks-charts/stable/aws-load-balancer-controller//crds?ref=master"
        const awslbcCrdsUrl = 'https://raw.githubusercontent.com/aws/eks-charts/master/stable/aws-load-balancer-controller/crds/crds.yaml';
        const awslbcCrdsYaml = request('GET', awslbcCrdsUrl).getBody('utf8');
        const awslbcCrdsObjects = yaml.loadAll(awslbcCrdsYaml);
        new eks.KubernetesManifest(this, 'alb-custom-resource-definition', {
            cluster: props.cluster,
            manifest: awslbcCrdsObjects,
            overwrite: true,
            prune: true,
        });
        // The following is equivalent to:
        // helm repo add eks https://aws.github.io/eks-charts
        // helm install aws-load-balancer-controller eks/aws-load-balancer-controller -n kube-system --set clusterName=<cluster-name> --set serviceAccount.create=false --set serviceAccount.name=aws-load-balancer-controller
        this.chart = new eks.HelmChart(scope, 'alb-ingress-controller', {
            cluster: props.cluster,
            wait: true,
            chart: 'aws-load-balancer-controller',
            repository: 'https://aws.github.io/eks-charts',
            // Note: the chart version 1.2.0 will install version 2.2.0 of the Helm Chart
            // https://github.com/aws/eks-charts/blob/master/stable/aws-load-balancer-controller/Chart.yaml
            version: '1.2.0',
            namespace: 'kube-system',
            values: {
                clusterName: props.cluster.clusterName,
                serviceAccount: {
                    create: false,
                    name: 'aws-load-balancer-controller',
                },
            },
        });
    }
}
exports.AwsLoadBalancerController = AwsLoadBalancerController;
//# sourceMappingURL=data:application/json;base64,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