"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExternalDns = void 0;
const eks = require("@aws-cdk/aws-eks");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
class ExternalDns extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Namespace in which external-dns is deployed
         */
        const ns = 'external-dns';
        const externalDnsNamespace = props.cluster.addManifest('external-dns-ns', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: ns,
            },
        });
        /**
         * Service account for external-dns
         */
        const externalDNSServiceAccount = props.cluster.addServiceAccount(ns, {
            name: ns,
            namespace: ns,
        });
        /**
         * Policies that will allow
         */
        const r53ListPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'route53:ListHostedZones',
                'route53:ListResourceRecordSets',
            ],
            resources: ['*'],
        });
        const r53UpdateRecordPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'route53:ChangeResourceRecordSets',
            ],
            resources: [props.hostedZone.hostedZoneArn],
        });
        externalDNSServiceAccount.addToPrincipalPolicy(r53ListPolicy);
        externalDNSServiceAccount.addToPrincipalPolicy(r53UpdateRecordPolicy);
        /**
         * Bitnami Helm chart for external-dns
         */
        const externalDnsChart = new eks.HelmChart(scope, 'external-dns', {
            cluster: props.cluster,
            namespace: ns,
            wait: true,
            release: 'external-dns',
            chart: 'external-dns',
            repository: 'https://charts.bitnami.com/bitnami',
            values: {
                serviceAccount: {
                    create: false,
                    name: ns,
                },
                namespace: 'external-dns',
                provider: 'aws',
                // podAnnotations: {
                //   'app.kubernetes.io/managed-by': 'Helm',
                // },
                aws: {
                    zoneType: 'public',
                },
                txtOwnerId: props.hostedZone.hostedZoneId,
                domainFilters: [
                    props.hostedZone.zoneName,
                ],
            },
        });
        /**
         * The namespace that we deploy this chart into must be deployed before deploying the chart and service account
         */
        externalDnsChart.node.addDependency(externalDnsNamespace);
        externalDNSServiceAccount.node.addDependency(externalDnsNamespace);
    }
}
exports.ExternalDns = ExternalDns;
//# sourceMappingURL=data:application/json;base64,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