"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Irsa = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * Pod Role setup with IRSA (IAM Roles for Service Accounts)
 * https://docs.aws.amazon.com/eks/latest/userguide/iam-roles-for-service-accounts-technical-overview.html
 */
class Irsa extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const POD_SERVICE_ACCOUNT_NAME = 'pod-service-account';
        const oidcProviderId = props.cluster.openIdConnectProvider.openIdConnectProviderIssuer;
        /**
         * The Principal that will assume the podRole
         */
        const federatedPrincipal = new iam.FederatedPrincipal(props.cluster.openIdConnectProvider.openIdConnectProviderArn, {
            StringEquals: new cdk.CfnJson(scope, 'FederatedPrincipalCondition', {
                value: {
                    [`${oidcProviderId}:aud`]: 'sts.amazonaws.com',
                    [`${oidcProviderId}:sub`]: `system:serviceaccount:app:${POD_SERVICE_ACCOUNT_NAME}`,
                },
            }),
        }, 'sts:AssumeRoleWithWebIdentity');
        /**
         * Setup a new IAM Role that will be used by Pods to access Secrets Manager, S3, etc.
         */
        this.podRole = new iam.Role(scope, 'podRole', {
            assumedBy: federatedPrincipal,
        });
        const podServiceAccount = {
            apiVersion: 'v1',
            kind: 'ServiceAccount',
            metadata: {
                name: POD_SERVICE_ACCOUNT_NAME,
                namespace: 'app',
                annotations: {
                    'eks.amazonaws.com/role-arn': this.podRole.roleArn,
                },
            },
        };
        /**
         * Apply the service account manfiest that will be used by pods running the application
         */
        this.chart = props.cluster.addManifest('pod-service-account', podServiceAccount);
    }
}
exports.Irsa = Irsa;
//# sourceMappingURL=data:application/json;base64,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