"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const stream_1 = require("stream");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const code_artifact_lambda_shared_1 = require("../../../backend/shared/code-artifact.lambda-shared");
jest.mock('child_process');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    done();
});
test('logInWithCodeArtifact', async () => {
    // GIVEN
    const protoEndpoint = '//fake.npm.endpoint';
    const endpoint = `https:${protoEndpoint}`;
    const domain = 'fake-domain-name';
    const domainOwner = '123456789012';
    const apiEndpoint = 'https://fake.codeartifact.api.endpoint';
    const authorizationToken = crypto_1.randomBytes(64).toString('base64');
    AWSMock.mock('CodeArtifact', 'getAuthorizationToken', (param, cb) => {
        try {
            expect(param.domain).toBe(domain);
            expect(param.domainOwner).toBe(domainOwner);
            expect(param.durationSeconds).toBe(0);
        }
        catch (e) {
            return cb(e);
        }
        cb(null, { authorizationToken });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockSpawn = require('child_process').spawn;
    const configToSet = new Set([
        `registry=${endpoint}`,
        `${protoEndpoint}:_authToken=${authorizationToken}`,
        `${protoEndpoint}:always-auth=true`,
    ]);
    mockSpawn.mockImplementation((cmd, args, opts) => {
        let result = Promise.resolve(null);
        try {
            expect(cmd).toBe('npm');
            const [config, set, keyValuePair, ...rest] = args;
            expect(rest).toEqual([]);
            expect(config).toBe('config');
            expect(set).toBe('set');
            expect(configToSet).toContain(keyValuePair);
            configToSet.delete(keyValuePair); // Set, remove it from here...
            expect(opts.stdio).toEqual(['ignore', 'inherit', 'inherit']);
        }
        catch (e) {
            result = Promise.reject(e);
        }
        return new MockChildProcess(cmd, Array.from(args), result);
    });
    // THEN
    await expect(code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint }))
        .resolves.not.toThrowError();
    expect(Array.from(configToSet)).toEqual([]); // All config was set as expected.
});
class MockChildProcess extends stream_1.EventEmitter {
    constructor(spawnfile, spawnargs, promise) {
        super();
        this.spawnfile = spawnfile;
        this.spawnargs = spawnargs;
        this.stdin = null;
        this.stdout = null;
        this.stderr = null;
        this.stdio = [this.stdin, this.stdout, this.stderr, null, null];
        this.exitCode = 0;
        this.killed = false;
        this.signalCode = null;
        this.pid = -1; // Obviously fake
        promise.then(() => this.emit('close', this.exitCode, this.signalCode), (err) => this.emit('error', err));
    }
    get connected() {
        throw new Error('Not Implemented');
    }
    disconnect() {
        throw new Error('Not Implemented');
    }
    kill() {
        throw new Error('Not Implemented');
    }
    ref() {
        throw new Error('Not Implemented');
    }
    send() {
        throw new Error('Not Implemented');
    }
    unref() {
        throw new Error('Not Implemented');
    }
}
//# sourceMappingURL=data:application/json;base64,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