"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const zlib_1 = require("zlib");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws.s3().getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        }).promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        const tar = await gunzip(Buffer.from(tarball.Body));
        let dotJsii;
        let licenseText;
        let packageJson;
        try {
            const extracted = await new Promise((ok, ko) => {
                let dotJsiiBuffer;
                let licenseTextBuffer;
                let packageJsonData;
                const extractor = tar_stream_1.extract({ filenameEncoding: 'utf-8' })
                    .once('error', (reason) => {
                    ko(reason);
                })
                    .once('finish', () => {
                    if (dotJsiiBuffer == null) {
                        ko(new Error('No .jsii file found in tarball!'));
                    }
                    else if (packageJsonData == null) {
                        ko(new Error('No package.json file found in tarball!'));
                    }
                    else {
                        ok({ dotJsii: dotJsiiBuffer, licenseText: licenseTextBuffer, packageJson: packageJsonData });
                    }
                })
                    .on('entry', (headers, stream, next) => {
                    const chunks = new Array();
                    if (headers.name === 'package/.jsii') {
                        return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                            .once('error', ko)
                            .once('end', () => {
                            dotJsiiBuffer = Buffer.concat(chunks);
                            // Skip on next runLoop iteration so we avoid filling the stack.
                            setImmediate(next);
                        })
                            .resume();
                    }
                    else if (headers.name === 'package/package.json') {
                        return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                            .once('error', ko)
                            .once('end', () => {
                            packageJsonData = Buffer.concat(chunks);
                            // Skip on next runLoop iteration so we avoid filling the stack.
                            setImmediate(next);
                        })
                            .resume();
                    }
                    else if (isLicenseFile(headers.name)) {
                        return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                            .once('error', ko)
                            .once('end', () => {
                            licenseTextBuffer = Buffer.concat(chunks);
                            // Skip on next runLoop iteration so we avoid filling the stack.
                            setImmediate(next);
                        })
                            .resume();
                    }
                    // Skip on next runLoop iteration so we avoid filling the stack.
                    return setImmediate(next);
                });
                extractor.write(tar, (err) => {
                    if (err != null) {
                        ko(err);
                    }
                    extractor.end();
                });
            });
            dotJsii = extracted.dotJsii;
            licenseText = extracted.licenseText;
            packageJson = extracted.packageJson;
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        const metadata = { date: payload.time, licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8') };
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const { license, name, version } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense } = JSON.parse(packageJson.toString('utf-8'));
        if (packageJsonName !== packageName || packageJsonVersion !== packageVersion || packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws.s3().putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        }).promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws.stepFunctions().startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        }).promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
function gunzip(data) {
    const chunks = new Array();
    return new Promise((ok, ko) => zlib_1.createGunzip()
        .once('error', ko)
        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
        .once('end', () => ok(Buffer.concat(chunks)))
        .end(data));
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return possibleExtensions.has(ext.toLowerCase())
        && path_1.basename(fileName, ext).toUpperCase() === 'LICENSE';
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/ig, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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