"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const catalog_builder_1 = require("../catalog-builder");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const reprocess_all_1 = require("./reprocess-all");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT];
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function;
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], interval: core_1.Duration.seconds(30), maxAttempts: 5 })
            .addCatch(new aws_stepfunctions_1.Pass(this, 'Add to catalog.json failure', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, 'Add to catalog.json fault', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.ALL'] });
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => new tasks.LambdaInvoke(this, `Generate ${language} docs`, {
            lambdaFunction: new transliterator_1.Transliterator(this, `DocGen-${language}`, { ...props, language }).function,
            outputPath: '$.result',
            resultSelector: {
                result: {
                    'language': language,
                    'success.$': '$.Payload',
                },
            },
        }).addRetry({ interval: core_1.Duration.seconds(30) })
            .addCatch(new aws_stepfunctions_1.Pass(this, `Generate ${language} docs failure`, { parameters: { 'error.$': 'States.StringToJson($.Cause)', language } }), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, `Generate ${language} docs fault`, { parameters: { 'error.$': '$.Cause', language } }), { errors: ['States.ALL'] })))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineType: aws_stepfunctions_1.StateMachineType.STANDARD,
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // This function is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the back-end.
        this.reprocessAllFunction = new reprocess_all_1.ReprocessAll(this, 'ReprocessAll', {
            description: '[ConstructHub/ReprocessAll] Reprocess all package versions through the backend',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.bucket.grantRead(this.reprocessAllFunction);
        this.stateMachine.grantStartExecution(this.reprocessAllFunction);
    }
}
exports.Orchestration = Orchestration;
//# sourceMappingURL=data:application/json;base64,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