"""The module that defines the ``ExtendedBulkCourseRegistrationLink`` model.

SPDX-License-Identifier: AGPL-3.0-only OR BSD-3-Clause-Clear
"""

import typing as t
from dataclasses import dataclass, field

import cg_request_args as rqa

from .. import parsers
from ..utils import to_dict
from .bulk_course_registration_link import BulkCourseRegistrationLink
from .course import Course


@dataclass
class ExtendedBulkCourseRegistrationLink(BulkCourseRegistrationLink):
    """The extended variant of a course registration link."""

    #: The course this link will enroll users in
    course: "Course"

    raw_data: t.Optional[t.Dict[str, t.Any]] = field(init=False, repr=False)

    data_parser: t.ClassVar = rqa.Lazy(
        lambda: BulkCourseRegistrationLink.data_parser.parser.combine(
            rqa.FixedMapping(
                rqa.RequiredArgument(
                    "course",
                    parsers.ParserFor.make(Course),
                    doc="The course this link will enroll users in",
                ),
            )
        ).use_readable_describe(True)
    )

    def to_dict(self) -> t.Dict[str, t.Any]:
        res: t.Dict[str, t.Any] = {
            "course": to_dict(self.course),
            "id": to_dict(self.id),
            "type": to_dict(self.type),
            "expiration_date": to_dict(self.expiration_date),
            "role": to_dict(self.role),
        }
        return res

    @classmethod
    def from_dict(
        cls: t.Type["ExtendedBulkCourseRegistrationLink"],
        d: t.Dict[str, t.Any],
    ) -> "ExtendedBulkCourseRegistrationLink":
        parsed = cls.data_parser.try_parse(d)

        res = cls(
            course=parsed.course,
            id=parsed.id,
            type=parsed.type,
            expiration_date=parsed.expiration_date,
            role=parsed.role,
        )
        res.raw_data = d
        return res


import os

if os.getenv("CG_GENERATING_DOCS", "False").lower() in ("", "true"):
    # fmt: off
    import datetime

    from .course_role import CourseRole

    # fmt: on
