"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts"],
            },
        });
        this.targetOverrides = {};
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.instantiationOrderSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project, targets) {
        const _package = project.tryFindObjectFile("package.json");
        _package?.addOverride("nx", {
            targets: targets,
        });
        !_package && (this.targetOverrides[project.outdir] = targets);
    }
    // Remove this hack once subProjects is made public in Projen
    get instantiationOrderSubProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects;
    }
    get subProjects() {
        return this.instantiationOrderSubProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.wirePythonDependencies();
        this.synthesizeNonNodePackageJson();
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        const subProjectPackages = [];
        this.subProjects.forEach((subProject) => {
            if (isNodeProject(subProject)) {
                const subNodeProject = subProject;
                subProjectPackages.push(subNodeProject.package);
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => { };
            }
        });
        super.synth();
        // Force workspace install deps if any node subproject package has change, unless the workspace changed
        if (
        // @ts-ignore - `file` is private
        this.package.file.changed !== true &&
            // @ts-ignore - `file` is private
            subProjectPackages.find((pkg) => pkg.file.changed === true)) {
            // @ts-ignore - `installDependencies` is private
            this.package.installDependencies();
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            (manifest.nx = this.targetOverrides[subProject.outdir]
                ? { targets: this.targetOverrides[subProject.outdir] }
                : undefined),
                (manifest.name = subProject.name);
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.disableNoHoistBundled !== true) {
            const noHoistBundled = this.subProjects.flatMap((sub) => {
                if (sub instanceof javascript_1.NodeProject) {
                    return sub.deps.all
                        .filter((dep) => dep.type === projen_1.DependencyType.BUNDLED)
                        .flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.workspacePackages,
                nohoist: noHoist,
            });
        }
    }
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    wirePythonDependencies() {
        // Find any python projects
        const pythonProjects = this.subProjects.filter((project) => project instanceof python_1.PythonProject);
        if (pythonProjects.length === 0) {
            // Nothing to do for no python projects
            return;
        }
        // Move all install commands to install-py so that they are not installed in parallel by the monorepo package manager.
        // eg yarn install will run the install task for all packages in parallel which can lead to conflicts for python.
        pythonProjects.forEach((pythonProject) => {
            const installPyTask = pythonProject.tasks.tryFind("install-py") ??
                pythonProject.addTask("install-py");
            const installTask = pythonProject.tasks.tryFind("install");
            (installTask?.steps || []).forEach((step) => {
                this.copyStepIntoTask(step, installPyTask, pythonProject);
            });
            installTask?.reset();
        });
        // Add an install task to the monorepo to include running the install-py command serially to avoid conflicting writes
        // to a shared virtual env. This is also managed by nx so that installs occur in dependency order.
        const monorepoInstallTask = this.tasks.tryFind("install") ?? this.addTask("install");
        monorepoInstallTask.exec(`npx nx run-many --target install-py --projects ${pythonProjects
            .map((project) => project.name)
            .join(",")} --parallel=1`);
        // Update the nx.json to ensure that install-py follows dependency order
        this.nxJson.addOverride("targetDependencies.install-py", [
            {
                target: "install-py",
                projects: "dependencies",
            },
        ]);
    }
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    copyStepIntoTask(step, task, project) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = project.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.nx_monorepo.NxMonorepoProject", version: "0.12.28" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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