"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontOriginAccessIdentity = exports.CloudFrontDistributionForMediaStore = exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const cdk = require("aws-cdk-lib");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    utils_1.addCfnSuppressRules(cfDistribution, [
        {
            id: 'W70',
            reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
        }
    ]);
    return cfDistribution;
}
// Cloudfront function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultCloudfrontFunction(scope) {
    // generate a stable unique id for the cloudfront function and use it
    // both for the function name and the logical id of the function so if
    // it is changed the function will be recreated.
    // see https://github.com/aws/aws-cdk/issues/15523
    const functionId = `SetHttpSecurityHeaders${scope.node.addr}`;
    return new cloudfront.Function(scope, "SetHttpSecurityHeaders", {
        functionName: functionId,
        code: cloudfront.FunctionCode.fromInline("function handler(event) { var response = event.response; \
      var headers = response.headers; \
      headers['strict-transport-security'] = { value: 'max-age=63072000; includeSubdomains; preload'}; \
      headers['content-security-policy'] = { value: \"default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self'; object-src 'none'\"}; \
      headers['x-content-type-options'] = { value: 'nosniff'}; \
      headers['x-frame-options'] = {value: 'DENY'}; \
      headers['x-xss-protection'] = {value: '1; mode=block'}; \
      return response; \
    }")
    });
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, httpSecurityHeaders, cloudfrontFunction);
    const cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders = true, originPath, cloudFrontLoggingBucketProps) {
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, httpSecurityHeaders, originPath, cloudfrontFunction);
    const cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    // the lack of a bucketPolicy means the bucket was imported from outside the stack so the lack of cfn_nag suppression is not an issue
    if (bucketPolicy) {
        utils_1.addCfnSuppressRules(bucketPolicy, [
            {
                id: 'F16',
                reason: `Public website bucket policy requires a wildcard principal`
            }
        ]);
    }
    return [cfDistribution, cloudfrontFunction, loggingBucket];
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
function CloudFrontDistributionForMediaStore(scope, mediaStoreContainer, cloudFrontDistributionProps, httpSecurityHeaders = true, cloudFrontLoggingBucketProps) {
    let originRequestPolicy;
    const loggingBucket = getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps);
    if (cloudFrontDistributionProps
        && cloudFrontDistributionProps.defaultBehavior
        && cloudFrontDistributionProps.defaultBehavior.originRequestPolicy) {
        originRequestPolicy = cloudFrontDistributionProps.defaultBehavior.originRequestPolicy;
    }
    else {
        const originRequestPolicyProps = {
            headerBehavior: {
                behavior: 'whitelist',
                headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            queryStringBehavior: {
                behavior: 'all'
            },
            cookieBehavior: {
                behavior: 'none'
            },
            comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            originRequestPolicyName: `${cdk.Aws.STACK_NAME}-${cdk.Aws.REGION}-CloudFrontDistributionForMediaStore`
        };
        originRequestPolicy = new cloudfront.OriginRequestPolicy(scope, 'CloudfrontOriginRequestPolicy', originRequestPolicyProps);
    }
    const cloudfrontFunction = getCloudfrontFunction(httpSecurityHeaders, scope);
    const defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontDisributionForMediaStoreProps(mediaStoreContainer, loggingBucket, originRequestPolicy, httpSecurityHeaders, cloudFrontDistributionProps?.customHeaders, cloudfrontFunction);
    let cfprops;
    cfprops = utils_1.consolidateProps(defaultprops, cloudFrontDistributionProps);
    // Create the CloudFront Distribution
    const cfDistribution = new cloudfront.Distribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, loggingBucket, originRequestPolicy, cloudfrontFunction];
}
exports.CloudFrontDistributionForMediaStore = CloudFrontDistributionForMediaStore;
function CloudFrontOriginAccessIdentity(scope, comment) {
    return new cloudfront.OriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        comment: comment ? comment : `access-identity-${cdk.Aws.REGION}-${cdk.Aws.STACK_NAME}`
    });
}
exports.CloudFrontOriginAccessIdentity = CloudFrontOriginAccessIdentity;
function getLoggingBucket(cloudFrontDistributionProps, scope, cloudFrontLoggingBucketProps) {
    const isLoggingDisabled = cloudFrontDistributionProps?.enableLogging === false;
    const userSuppliedLogBucket = cloudFrontDistributionProps?.logBucket;
    if (userSuppliedLogBucket && cloudFrontLoggingBucketProps) {
        throw Error('Either cloudFrontDistributionProps.logBucket or cloudFrontLoggingBucketProps can be set.');
    }
    return isLoggingDisabled
        ? undefined
        : userSuppliedLogBucket ?? s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket', cloudFrontLoggingBucketProps ? utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), cloudFrontLoggingBucketProps) : s3_bucket_defaults_1.DefaultS3Props());
}
function getCloudfrontFunction(httpSecurityHeaders, scope) {
    return httpSecurityHeaders ? defaultCloudfrontFunction(scope) : undefined;
}
//# sourceMappingURL=data:application/json;base64,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