"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGlueDatabase = exports.createGlueTable = exports.createGlueJobRole = exports.deployGlueJob = exports.buildGlueJob = exports.SinkStoreType = void 0;
const glue = require("aws-cdk-lib/aws-glue");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
const utils_1 = require("./utils");
/**
 * Enumeration of data store types that could include S3, DynamoDB, DocumentDB, RDS or Redshift. Current
 * construct implementation only supports S3, but potential to add other output types in the future
 */
var SinkStoreType;
(function (SinkStoreType) {
    SinkStoreType["S3"] = "S3";
})(SinkStoreType = exports.SinkStoreType || (exports.SinkStoreType = {}));
function buildGlueJob(scope, props) {
    if (!props.existingCfnJob) {
        if (props.glueJobProps) {
            if (props.glueJobProps.glueVersion === '2.0' && props.glueJobProps.maxCapacity) {
                throw Error('Cannot set "MaxCapacity" with GlueVersion 2.0 or higher. Use "NumberOfWorkers" and "WorkerType". ' +
                    'Refer the API documentation https://docs.aws.amazon.com/glue/latest/webapi/API_Job.html for more details');
            }
            if (props.glueJobProps.maxCapacity && (props.glueJobProps.numberOfWorkers || props.glueJobProps.workerType)) {
                throw Error('Cannot set MaxCapacity and "WorkerType" or  "NumberOfWorkers". If using glueVersion 2.0 or beyond, ' +
                    'it is recommended to use "WorkerType" or  "NumberOfWorkers"');
            }
            return deployGlueJob(scope, props.glueJobProps, props.database, props.table, props.outputDataStore, props.etlCodeAsset);
        }
        else {
            throw Error('Either glueJobProps or existingCfnJob is required');
        }
    }
    else {
        return [props.existingCfnJob, aws_iam_1.Role.fromRoleArn(scope, 'ExistingRole', props.existingCfnJob.role)];
    }
}
exports.buildGlueJob = buildGlueJob;
function deployGlueJob(scope, glueJobProps, database, table, outputDataStore, etlCodeAsset) {
    let _glueSecurityConfigName;
    if (glueJobProps.securityConfiguration === undefined) {
        _glueSecurityConfigName = 'ETLJobSecurityConfig';
        const _glueKMSKey = `arn:${aws_cdk_lib_1.Aws.PARTITION}:kms:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:alias/aws/glue`;
        new glue.CfnSecurityConfiguration(scope, 'GlueSecurityConfig', {
            name: _glueSecurityConfigName,
            encryptionConfiguration: {
                jobBookmarksEncryption: {
                    jobBookmarksEncryptionMode: 'CSE-KMS',
                    kmsKeyArn: _glueKMSKey
                },
                s3Encryptions: [{
                        s3EncryptionMode: 'SSE-S3'
                    }]
            }
        });
    }
    else {
        _glueSecurityConfigName = glueJobProps.securityConfiguration;
    }
    const _glueJobPolicy = new aws_iam_1.Policy(scope, 'LogPolicy', {
        statements: [
            new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/*`]
            })
        ]
    });
    let _jobRole;
    if (glueJobProps.role) {
        _jobRole = aws_iam_1.Role.fromRoleArn(scope, 'JobRole', glueJobProps.role);
    }
    else {
        _jobRole = defaults.createGlueJobRole(scope);
    }
    _glueJobPolicy.attachToRole(_jobRole);
    let _outputLocation;
    if (outputDataStore !== undefined && outputDataStore.datastoreType === SinkStoreType.S3) {
        if (outputDataStore.existingS3OutputBucket !== undefined) {
            _outputLocation = [outputDataStore.existingS3OutputBucket, undefined];
        }
        else {
            _outputLocation = defaults.buildS3Bucket(scope, { bucketProps: outputDataStore.outputBucketProps });
        }
    }
    else {
        _outputLocation = defaults.buildS3Bucket(scope, {});
    }
    _outputLocation[0].grantReadWrite(_jobRole);
    const _jobArgumentsList = {
        "--enable-metrics": true,
        "--enable-continuous-cloudwatch-log": true,
        "--database_name": database.ref,
        "--table_name": table.ref,
        ...((outputDataStore === undefined || (outputDataStore && outputDataStore.datastoreType === SinkStoreType.S3)) &&
            { '--output_path': `s3a://${_outputLocation[0].bucketName}/output/` }),
        ...glueJobProps.defaultArguments
    };
    const _newGlueJobProps = utils_1.overrideProps(defaults.DefaultGlueJobProps(_jobRole, glueJobProps, _glueSecurityConfigName, _jobArgumentsList, etlCodeAsset), glueJobProps);
    if (etlCodeAsset) {
        etlCodeAsset.grantRead(_jobRole);
    }
    else {
        // create CDK Bucket instance from S3 url and grant read access to Glue Job's service principal
        if (isJobCommandProperty(_newGlueJobProps.command)) {
            if (!_newGlueJobProps.command.scriptLocation) {
                throw Error('Script location has to be provided as an s3 Url location. Script location cannot be empty');
            }
            const _scriptLocation = _newGlueJobProps.command.scriptLocation;
            const _scriptBucketLocation = aws_s3_1.Bucket.fromBucketArn(scope, 'ScriptLocaiton', getS3ArnfromS3Url(_scriptLocation));
            _scriptBucketLocation.grantRead(_jobRole);
        }
    }
    const _glueJob = new glue.CfnJob(scope, 'KinesisETLJob', _newGlueJobProps);
    return [_glueJob, _jobRole, _outputLocation];
}
exports.deployGlueJob = deployGlueJob;
/**
 * This is a helper method to create the Role required for the Glue Job. If a role is already created then this
 * method is not required to be called.
 *
 * @param scope - The AWS Construct under which the role is to be created
 */
function createGlueJobRole(scope) {
    return new aws_iam_1.Role(scope, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        description: 'Service role that Glue custom ETL jobs will assume for exeuction',
    });
}
exports.createGlueJobRole = createGlueJobRole;
/**
 * This method creates an AWS Glue table. The method is called when an existing Glue table is not provided
 */
function createGlueTable(scope, database, tableProps, fieldSchema, sourceType, parameters) {
    return defaults.DefaultGlueTable(scope, tableProps !== undefined ? tableProps :
        defaults.DefaultGlueTableProps(database, fieldSchema, sourceType, parameters));
}
exports.createGlueTable = createGlueTable;
/**
 * This method creates an AWS Glue database. The method is only called with an existing Glue database type is not provided.
 * The method uses the user provided props to override the defaul props for the Glue database
 *
 * @param scope
 * @param databaseProps
 */
function createGlueDatabase(scope, databaseProps) {
    const _mergedDBProps = (databaseProps !== undefined) ? utils_1.overrideProps(defaults.DefaultGlueDatabaseProps(), databaseProps) :
        defaults.DefaultGlueDatabaseProps();
    return defaults.DefaultGlueDatabase(scope, _mergedDBProps);
}
exports.createGlueDatabase = createGlueDatabase;
/**
 * A utility method to generate the S3 Arn from an S3 Url.
 *
 * @param s3Url
 */
function getS3ArnfromS3Url(s3Url) {
    if (s3Url && s3Url.startsWith('s3://')) {
        const splitString = s3Url.slice('s3://'.length);
        return `arn:${aws_cdk_lib_1.Aws.PARTITION}:s3:::${splitString}`;
    }
    else {
        throw Error(`Received S3URL as ${s3Url}. The S3 url string does not begin with s3://. This is not a standard S3 url`);
    }
}
/**
 * A utility method to type check CfnJob.JobCommandProperty type. For the construct to work for streaming ETL from Kinesis Data
 * Streams, all three attributes of the JobCommandProperty are required, even though they may be optional for other use cases.
 *
 * @param command
 */
function isJobCommandProperty(command) {
    if (command.name &&
        command.pythonVersion &&
        command.scriptLocation) {
        return true;
    }
    else {
        defaults.printWarning('command not of type JobCommandProperty type');
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,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