"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.flagOverriddenDefaults = void 0;
const deepdiff = require("deep-diff");
const utils_1 = require("./utils");
/**
 * Emits a warning to the console when a prescriptive default value is overridden by the user.
 * @param {object} defaultProps the prescriptive defaults for the pattern.
 * @param {object} userProps the properties provided by the user, to be compared against the defaultProps.
 */
function flagOverriddenDefaults(defaultProps, userProps) {
    // Compare the properties and return any overrides
    const overrides = findOverrides(defaultProps, userProps);
    // Emit a warning for each override
    for (let i = 0; i < ((overrides !== undefined) ? overrides.length : 0); i++) {
        const e = Object.assign(overrides[i]);
        // Determine appropriate logging granularity
        const valuesAreReadable = (checkReadability(e.lhs) &&
            checkReadability(e.rhs));
        // Format the path for readability
        const path = formatOverridePath(e.path);
        // Output
        const details = (valuesAreReadable) ? ` Default value: '${e.lhs}'. You provided: '${e.rhs}'.` : '';
        utils_1.printWarning(`An override has been provided for the property: ${path}.` + details);
    }
}
exports.flagOverriddenDefaults = flagOverriddenDefaults;
/** The prefilter function returns true for any filtered path/key that should be excluded from the diff check.
 * Any Construct Props using cdk.Duration type is not properly handled by
 * 'deep-diff' library, whenever it encounters a Duration object, it throws the exception
 * 'Argument to Intrinsic must be a plain value object', so such props are excluded from the diff check.
 */
function _prefilter(_path, _key) {
    const prefilters = ['maxRecordAge', 'expiration', 'transitionAfter', 'destination', 'timeout', 'period'];
    if (prefilters.indexOf(_key) >= 0) {
        return true;
    }
    return false;
}
/**
 * Performs a diff check of the userProps against the defaultProps to detect overridden properties.
 * @param {object} defaultProps the prescriptive defaults for the pattern.
 * @param {object} userProps the properties provided by the user, to be compared against the defaultProps.
 * @return {Array} an array containing the overridden values.
 */
function findOverrides(defaultProps, userProps) {
    const diff = deepdiff.diff(defaultProps, userProps, _prefilter);
    // Filter the results
    return (diff !== undefined) ? diff?.filter((e) => (e.kind === 'E' && // only return overrides
        !e.path?.includes('node') && // stop traversing once the object graph hits the node
        !e.path?.includes('bind') // stop traversing once the object graph hits internal functions
    )) : [];
}
/**
 * Converts the path object from the deep-diff module to a user-readable, bracket notation format.
 * @param {string | string[]} path either a string value or an array of strings.
 * @return {string} the formatted override path.
 */
function formatOverridePath(path) {
    return (path !== undefined && path.length > 1) ? path.toString()
        .replace(/,/g, '][')
        .replace(/\]{1}/, '')
        .replace(/$/, ']') : path;
}
/**
 * Check the readability of an input value and, in the context of the override warning service, return true if it
 * meets the established criteria. This function is used to determine whether more-detailed log output can be given.
 * @param {any} value input to be evaluated against the given criteria.
 * @return {boolean} true if the value meets the given criteria.
 * @return {boolean} false if the value does not meet the given criteria.
 */
function checkReadability(value) {
    return (typeof (value) === 'string' && // strings only
        !value.includes('${Token[') && // no circular refs
        value !== '' // no empty strings
    );
}
//# sourceMappingURL=data:application/json;base64,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