"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
const cdk = require("aws-cdk-lib");
const smDefaults = require("./step-function-defaults");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("aws-cdk-lib/aws-iam");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
/**
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps, logGroupProps) {
    let logGroup;
    let _smProps;
    // If they sent a logGroup in stateMachineProps
    if (stateMachineProps.logs?.destination) {
        logGroup = stateMachineProps.logs?.destination;
        _smProps = stateMachineProps;
    }
    else {
        // Three possibilities
        // 1) logGroupProps not provided - create logGroupProps with just logGroupName
        // 2) logGroupProps provided with no logGroupName - override logGroupProps.logGroupName
        // 3) logGroupProps provided with logGroupName - pass unaltered logGroupProps
        let consolidatedLogGroupProps = logGroupProps;
        if (!consolidatedLogGroupProps) {
            consolidatedLogGroupProps = {};
        }
        if (!consolidatedLogGroupProps?.logGroupName) {
            const logGroupPrefix = '/aws/vendedlogs/states/';
            const maxResourceNameLength = 255 - logGroupPrefix.length;
            const nameParts = [
                cdk.Stack.of(scope).stackName,
                scope.node.id,
                'StateMachineLog' // Literal string for log group name portion
            ];
            const logGroupName = logGroupPrefix + utils_1.generateResourceName(nameParts, maxResourceNameLength);
            consolidatedLogGroupProps = utils_1.overrideProps(consolidatedLogGroupProps, { logGroupName });
        }
        // Create new Cloudwatch log group for Step function State Machine
        logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'StateMachineLogGroup', consolidatedLogGroupProps);
        // Override the defaults with the user provided props
        _smProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    }
    // Override the Cloudwatch permissions to make it more fine grained
    const _sm = new sfn.StateMachine(scope, 'StateMachine', _smProps);
    const role = _sm.node.findChild('Role');
    const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
    // Reduce the scope of actions for the existing DefaultPolicy
    cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
        "logs:CreateLogDelivery",
        'logs:GetLogDelivery',
        'logs:UpdateLogDelivery',
        'logs:DeleteLogDelivery',
        'logs:ListLogDeliveries'
    ]);
    // Override Cfn Nag warning W12: IAM policy should not allow * resource
    utils_1.addCfnSuppressRules(cfnDefaultPolicy, [
        {
            id: 'W12',
            reason: `The 'LogDelivery' actions do not support resource-level authorizations`
        }
    ]);
    // Add a new policy with logging permissions for the given cloudwatch log group
    _sm.addToRolePolicy(new iam.PolicyStatement({
        actions: [
            'logs:PutResourcePolicy',
            'logs:DescribeResourcePolicies',
            'logs:DescribeLogGroups'
        ],
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
    }));
    return [_sm, logGroup];
}
exports.buildStateMachine = buildStateMachine;
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed({
            statistic: 'Sum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled({
            statistic: 'Sum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted({
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,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