"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("../index");
const aws_lambda_event_sources_1 = require("aws-cdk-lib/aws-lambda-event-sources");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
require("@aws-cdk/assert/jest");
const aws_cdk_lib_1 = require("aws-cdk-lib");
test('test DynamoEventSourceProps with defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = defaults.DynamoEventSourceProps(stack);
    expect(props.onFailure).toBeInstanceOf(aws_lambda_event_sources_1.SqsDlq);
    expect(props.startingPosition).toEqual("TRIM_HORIZON");
    expect(props.bisectBatchOnError).toEqual(true);
    expect(props.retryAttempts).toEqual(500);
    expect(props.maxRecordAge).toEqual(aws_cdk_lib_1.Duration.hours(24));
});
test('test DynamoEventSourceProps with deployDeadLetterQueue: false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = defaults.DynamoEventSourceProps(stack, {
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        startingPosition: "TRIM_HORIZON",
        bisectBatchOnError: true,
        maxRecordAge: aws_cdk_lib_1.Duration.hours(24),
        retryAttempts: 500
    });
});
test('test DynamoEventSourceProps override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const myProps = {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 5,
        retryAttempts: 3
    };
    const props = defaults.DynamoEventSourceProps(stack, {
        eventSourceProps: myProps,
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        batchSize: 5,
        startingPosition: "LATEST",
        maxRecordAge: aws_cdk_lib_1.Duration.hours(24),
        bisectBatchOnError: true,
        retryAttempts: 3
    });
});
test('test S3EventSourceProps w/ default props', () => {
    const props = defaults.S3EventSourceProps();
    expect(props).toEqual({
        events: ["s3:ObjectCreated:*"]
    });
});
test('test S3EventSourceProps w/ user props', () => {
    const s3EventSourceProps = {
        events: [
            s3.EventType.OBJECT_CREATED_POST
        ]
    };
    const props = defaults.S3EventSourceProps(s3EventSourceProps);
    expect(props).toEqual({
        events: ["s3:ObjectCreated:Post"]
    });
});
test('test KinesisEventSourceProps with defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = defaults.KinesisEventSourceProps(stack);
    expect(props.onFailure).toBeInstanceOf(aws_lambda_event_sources_1.SqsDlq);
    expect(props.startingPosition).toEqual("TRIM_HORIZON");
    expect(props.bisectBatchOnError).toEqual(true);
    expect(props.retryAttempts).toEqual(500);
    expect(props.maxRecordAge).toEqual(aws_cdk_lib_1.Duration.hours(24));
});
test('test KinesisEventSourceProps with deployDeadLetterQueue: false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = defaults.KinesisEventSourceProps(stack, {
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        startingPosition: "TRIM_HORIZON",
        bisectBatchOnError: true,
        maxRecordAge: aws_cdk_lib_1.Duration.hours(24),
        retryAttempts: 500
    });
});
test('test KinesisEventSourceProps override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const myProps = {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 5,
        retryAttempts: 3
    };
    const props = defaults.KinesisEventSourceProps(stack, {
        eventSourceProps: myProps,
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        batchSize: 5,
        startingPosition: "LATEST",
        maxRecordAge: aws_cdk_lib_1.Duration.hours(24),
        bisectBatchOnError: true,
        retryAttempts: 3
    });
});
test('test sqsDlqQueueProps override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.KinesisEventSourceProps(stack, {
        sqsDlqQueueProps: {
            queueName: 'hello-world',
            visibilityTimeout: aws_cdk_lib_1.Duration.seconds(50)
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "hello-world",
        VisibilityTimeout: 50
    });
});
//# sourceMappingURL=data:application/json;base64,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