"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const vpc_helper_1 = require("../lib/vpc-helper");
const vpc_defaults_1 = require("../lib/vpc-defaults");
// --------------------------------------------------------------
// Test minimal Isolated deployment with no properties
// --------------------------------------------------------------
test("Test minimal deployment with no properties", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultIsolatedVpcProps(),
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toHaveResource('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources('AWS::EC2::Subnet', 2);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources('AWS::EC2::InternetGateway', 0);
});
// --------------------------------------------------------------
// Test deployment w/ user provided custom properties
// --------------------------------------------------------------
test('Test deployment w/ user provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '172.168.0.0/19',
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '172.168.0.0/19',
        EnableDnsHostnames: false,
        EnableDnsSupport: false,
    });
});
// --------------------------------------------------------------
// Test deployment w/ construct provided custom properties
// --------------------------------------------------------------
test('Test deployment w/ construct provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: '172.168.0.0/19',
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '172.168.0.0/19',
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
// --------------------------------------------------------------
// Test deployment w/ construct and user provided custom properties
// --------------------------------------------------------------
test('Test deployment w/ construct and user provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '10.0.0.0/16',
        },
        constructVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '172.168.0.0/19',
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '172.168.0.0/19',
        EnableDnsHostnames: false,
        EnableDnsSupport: false,
    });
});
// --------------------------------------------------------------
// Test priority of default, user and construct properties
// --------------------------------------------------------------
test('Test deployment w/ construct and user provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const v = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        userVpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: '10.0.0.0/16',
        },
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            natGateways: 0,
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: 'isolated',
                    subnetType: ec2.SubnetType.PRIVATE_ISOLATED,
                },
            ],
        },
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, v, defaults.ServiceEndpointTypes.SQS);
    // Expect 2 isolated subnets (usual error condition is 2 public/2 private)
    expect(stack).toCountResources('AWS::EC2::Subnet', 2);
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 0);
});
// --------------------------------------------------------------
// Test deployment w/ existing VPC provided
// --------------------------------------------------------------
test('Test deployment w/ existing VPC provided', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testExistingVpc = defaults.getTestVpc(stack);
    const newVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
        existingVpc: testExistingVpc,
    });
    expect(newVpc).toBe(testExistingVpc);
});
// --------------------------------------------------------------
// Test adding Gateway Endpoint
// --------------------------------------------------------------
test('Test adding Gateway Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.DYNAMODB);
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SQS);
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    // Assertion
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Gateway',
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
    expect(stack).toCountResources('AWS::EC2::VPCEndpoint', 3);
});
// --------------------------------------------------------------
// Test adding Interface Endpoint
// --------------------------------------------------------------
test('Test adding Interface Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    // Assertion
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
});
// --------------------------------------------------------------
// Test adding SAGEMAKER_RUNTIME Interface Endpoint
// --------------------------------------------------------------
test('Test adding SAGEMAKER_RUNTIME Interface Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    // Assertion
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
});
// --------------------------------------------------------------
// Test adding a second Endpoint of same service
// --------------------------------------------------------------
test('Test adding a second Endpoint of same service', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.SNS);
    // Assertion
    expect(stack).toCountResources('AWS::EC2::VPCEndpoint', 1);
});
// --------------------------------------------------------------
// Test adding bad Endpoint
// --------------------------------------------------------------
test('Test adding bad Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    const app = () => {
        vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, 'string');
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test adding Events Interface Endpoint
// --------------------------------------------------------------
test('Test adding Events Interface Endpoint', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build VPC
    const testVpc = defaults.buildVpc(stack, {
        defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
    });
    vpc_helper_1.AddAwsServiceEndpoint(stack, testVpc, vpc_helper_1.ServiceEndpointTypes.EVENTS);
    // Assertion
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        VpcEndpointType: 'Interface',
    });
});
//# sourceMappingURL=data:application/json;base64,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