# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['xarray_dataclasses']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.19,<2.0', 'typing-extensions>=3.7,<4.0', 'xarray>=0.15,<0.16']

setup_kwargs = {
    'name': 'xarray-dataclasses',
    'version': '0.3.0',
    'description': 'xarray extension for DataArray and Dataset classes',
    'long_description': '# xarray-dataclasses\n\n[![PyPI](https://img.shields.io/pypi/v/xarray-dataclasses.svg?label=PyPI&style=flat-square)](https://pypi.org/project/xarray-dataclasses/)\n[![Python](https://img.shields.io/pypi/pyversions/xarray-dataclasses.svg?label=Python&color=yellow&style=flat-square)](https://pypi.org/project/xarray-dataclasses/)\n[![Test](https://img.shields.io/github/workflow/status/astropenguin/xarray-dataclasses/Test?logo=github&label=Test&style=flat-square)](https://github.com/astropenguin/xarray-dataclasses/actions)\n[![License](https://img.shields.io/badge/license-MIT-blue.svg?label=License&style=flat-square)](LICENSE)\n\nxarray extension for DataArray and Dataset classes\n\n\n## TL;DR\n\nxarray-dataclasses is a Python package for creating DataArray and Dataset classes in the same manner as [the Python\'s native dataclass].\nHere is an example code of what the package provides:\n\n```python\nfrom xarray_dataclasses import Coord, Data, dataarrayclass\n\n\n@dataarrayclass\nclass Image:\n    """DataArray that represents an image."""\n\n    data: Data[tuple[\'x\', \'y\'], float]\n    x: Coord[\'x\', int] = 0\n    y: Coord[\'y\', int] = 0\n\n\n# create a DataArray instance\nimage = Image.new([[0, 1], [2, 3]], x=[0, 1], y=[0, 1])\n\n\n# create a DataArray instance filled with ones\nones = Image.ones((2, 2), x=[0, 1], y=[0, 1])\n```\n\n### Features\n\n- DataArray or Dataset instances with fixed dimensions, data type, and coordinates can easily be created.\n- NumPy-like special functions such as ``ones()`` are provided as class methods.\n- 100% compatible with [the Python\'s native dataclass].\n- 100% compatible with static type check by [Pyright].\n\n### Installation\n\n```shell\n$ pip install xarray-dataclasses\n```\n\n\n## Introduction\n\n[xarray] is useful for handling labeled multi-dimensional data, but it is a bit troublesome to create a DataArray or Dataset instance with fixed dimensions, data type, or coordinates.\nFor example, let us think about the following specifications of DataArray instances:\n\n- Dimensions of data must be `(\'x\', \'y\')`.\n- Data type of data must be `float`.\n- Data type of dimensions must be `int`.\n- Default value of dimensions must be `0`.\n\nThen a function to create a spec-compliant DataArray instance is something like this:\n\n```python\nimport numpy as np\nimport xarray as xr\n\n\ndef spec_dataarray(data, x=None, y=None):\n    """Create a spec-comliant DataArray instance."""\n    data = np.array(data)\n\n    if x is None:\n        x = np.zeros(data.shape[0])\n    else:\n        x = np.array(x)\n\n    if y is None:\n        y = np.zeros(data.shape[1])\n    else:\n        y = np.array(y)\n\n    return xr.DataArray(\n        data=data.astype(float),\n        dims=(\'x\', \'y\'),\n        coords={\n            \'x\': (\'x\', x.astype(int)),\n            \'y\': (\'y\', y.astype(int)),\n        },\n    )\n\n\ndataarray = spec_dataarray([[0, 1], [2, 3]])\n```\n\nThe issues are (1) it is hard to figure out the specs from the code and (2) it is hard to reuse the code, for example, to add a new coordinate to the original specs.\n\n[xarray-dataclasses](#xarray-dataclasses) resolves them by defining the specs as a dataclass with dedicated type hints:\n\n```python\nfrom xarray_dataclasses import Coord, Data, dataarrayclass\n\n\n@dataarrayclass\nclass Specs:\n    data: Data[tuple[\'x\', \'y\'], float]\n    x: Coord[\'x\', int] = 0\n    y: Coord[\'y\', int] = 0\n\n\ndataarray = Specs.new([[0, 1], [2, 3]])\n```\n\nThe specs are now much easier to read:\nThe type hints, `Data[<dims>, <dtype>]` and `Coord[<dims>, <dtype>]`, have complete information of DataArray creation.\nThe default values are given as class variables.\n\nThe class decorator, `@dataarrayclass`, converts a class to [the Python\'s native dataclass] and add class methods such as `new()` to it.\nThe extension of the specs is then easy by class inheritance.\n\n## Basic usage\n\nxarray-dataclasses uses [the Python\'s native dataclass] (please learn how to use it before proceeding).\nData (or data variables), coordinates, attribute members, and name of a DataArray or Dataset instance are defined as dataclass fields with the following dedicated type hints.\n\n### Data\n\n`Data[<dims>, <dtype>]` specifies the field whose value will become the data of a DataArray instance or a member of the data variables of a Dataset instance.\nIt accepts two type variables, `<dims>` and `<dtype>`, for fixing dimensions and data type, respectively.\nFor example:\n\n| Type hint | Inferred dims | Inferred dtype |\n| --- | --- | --- |\n| `Data[\'x\', typing.Any]` | `(\'x\',)` | `None` |\n| `Data[\'x\', int]` | `(\'x\',)` | `numpy.dtype(\'int64\')` |\n| `Data[\'x\', float]` | `(\'x\',)` | `numpy.dtype(\'float64\')` |\n| `Data[tuple[\'x\', \'y\'], float]` | `(\'x\', \'y\')` | `numpy.dtype(\'float64\')` |\n\nNote: for Python 3.7 and 3.8, use `typing.Tuple[...]` instead of `tuple[...]`.\n\n### Coord\n\n`Coord[<dims>, <dtypes>]` specifies the field whose value will become a coordinate of a DataArray or Dataset instance.\nSimilar to `Data`, it accepts two type variables, `<dims>` and `<dtype>`, for fixing dimensions and data type, respectively.\n\n### Attr\n\n`Attr[<type>]` specifies the field whose value will become a member of the attributes (attrs) of a DataArray or Dataset instance.\nIt accepts a type variable, `<type>`, for specifying the type of the value.\nFor example:\n\n```python\n@dataarrayclass\nclass Specs:\n    units: Attr[str] = \'m/s\'  # equivalent to str\n```\n\n### Name\n\n`Name[<type>]` specifies the field whose value will become the name of a DataArray.\nIt accepts a type variable, `<type>`, for specifying the type of the value.\nFor example:\n\n```python\n@dataarrayclass\nclass Specs:\n    name: Name[str] = \'default\'  # equivalent to str\n```\n\n### DataArray class\n\nDataArray class is a dataclass that defines DataArray creation.\nFor example:\n\n```python\nfrom xarray_dataclasses import Attr, Coord, Data, Name, dataarrayclass\n\n\n@dataarrayclass\nclass Image:\n    """DataArray that represents an image."""\n\n    data: Data[tuple[\'x\', \'y\'], float]\n    x: Coord[\'x\', int] = 0\n    y: Coord[\'y\', int] = 0\n    dpi: Attr[int] = 300\n    name: Name[str] = \'default\'\n```\n\nwhere exactly one `Data`-typed field is allowed.\n`ValueError` is raised if more than two `Data`-type fields exist.\nA spec-compliant DataArray instance is created by a shorthand method, `new()`:\n\n```python\nImage.new([[0, 1], [2, 3]], x=[0, 1], y=[0, 1])\n\n<xarray.DataArray \'default\' (x: 2, y: 2)>\narray([[0., 1.],\n       [2., 3.]])\nCoordinates:\n  * x        (x) int64 0 1\n  * y        (y) int64 0 1\nAttributes:\n    dpi:      300\n```\n\nDataArray class has NumPy-like `empty()`, `zeros()`, `ones()`, `full()` methods:\n\n```python\nImage.ones((3, 3), dpi=200, name=\'flat\')\n\n<xarray.DataArray \'flat\' (x: 3, y: 3)>\narray([[1., 1., 1.],\n       [1., 1., 1.],\n       [1., 1., 1.]])\nCoordinates:\n  * x        (x) int64 0 0 0\n  * y        (y) int64 0 0 0\nAttributes:\n    dpi:      200\n```\n\n### Dataset class\n\nDataset class is a dataclass that defines Dataset creation.\nFor example:\n\n```python\nfrom xarray_dataclasses import Attr, Coord, Data, datasetclass\n\n\n@datasetclass\nclass RGBImage:\n    """Dataset that represents a three-color image."""\n\n    red: Data[tuple[\'x\', \'y\'], float]\n    green: Data[tuple[\'x\', \'y\'], float]\n    blue: Data[tuple[\'x\', \'y\'], float]\n    x: Coord[\'x\', int] = 0\n    y: Coord[\'y\', int] = 0\n    dpi: Attr[int] = 300\n```\n\nwhere multiple `Data`-typed fields are allowed.\nA spec-compliant Dataset instance is created by a shorthand method, `new()`:\n\n```python\nRGBImage.new(\n    [[0, 0], [0, 0]],  # red\n    [[1, 1], [1, 1]],  # green\n    [[2, 2], [2, 2]],  # blue\n)\n\n<xarray.Dataset>\nDimensions:  (x: 2, y: 2)\nCoordinates:\n  * x        (x) int64 0 0\n  * y        (y) int64 0 0\nData variables:\n    red      (x, y) float64 0.0 0.0 0.0 0.0\n    green    (x, y) float64 1.0 1.0 1.0 1.0\n    blue     (x, y) float64 2.0 2.0 2.0 2.0\nAttributes:\n    dpi:      300\n```\n\n\n<!-- References -->\n[Pyright]: https://github.com/microsoft/pyright\n[the Python\'s native dataclass]: https://docs.python.org/3/library/dataclasses.html\n[xarray]: https://xarray.pydata.org/en/stable/index.html\n',
    'author': 'Akio Taniguchi',
    'author_email': 'taniguchi@a.phys.nagoya-u.ac.jp',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/astropenguin/xarray-dataclasses/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
