"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
const DEFAULT_NAG_PACKS = [
    new cdk_nag_1.AwsSolutionsChecks({
        verbose: true,
        reports: true,
    }),
];
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
        this.nagPacks = props?.nagPacks ?? DEFAULT_NAG_PACKS;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNagApp", version: "0.14.20" };
class PDKNagAspect {
    constructor(app) {
        this.app = app;
    }
    visit(node) {
        this.app.nagPacks.forEach((nagPack) => nagPack.visit(node));
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            // Do Nothing
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            const stageName = aws_cdk_lib_1.Stage.of(stack)?.stageName;
            const stagePrefix = stageName && `${stageName}-`;
            let stackName = stack.stackName;
            stackName =
                stagePrefix && stackName.startsWith(stagePrefix)
                    ? `${stageName}/${stackName.slice(stagePrefix.length)}`
                    : stackName;
            return `${stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNag", version: "0.14.20" };
//# sourceMappingURL=data:application/json;base64,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