"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DenyListClient = void 0;
const aws_lambda_shared_1 = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const constants_1 = require("./constants");
/**
 * A client for working with the deny list.
 */
class DenyListClient {
    constructor() {
        this.bucketName = env_lambda_shared_1.requireEnv(constants_1.ENV_DENY_LIST_BUCKET_NAME);
        this.objectKey = env_lambda_shared_1.requireEnv(constants_1.ENV_DENY_LIST_OBJECT_KEY);
        this.s3 = aws_lambda_shared_1.s3();
    }
    /**
     * Creates a new client for accessing the deny list.
     */
    static async newClient() {
        const client = new DenyListClient();
        await client.init();
        return client;
    }
    /**
     * Downloads the deny list and stores it in memory.
     *
     * This must be called before `lookup()`.
     */
    async init() {
        if (this._map) {
            throw new Error('init() cannot be called twice');
        }
        this._map = {};
        try {
            const params = {
                Bucket: this.bucketName,
                Key: this.objectKey,
            };
            const { Body: body } = await this.s3.getObject(params).promise();
            if (!body) {
                console.log(`WARNING: deny list body is empty at ${this.bucketName}/${this.objectKey}`);
                return;
            }
            const contents = body.toString('utf-8');
            // an empty string is a valid (empty) deny list
            if (contents.length === 0) {
                return;
            }
            const data = JSON.parse(contents);
            if (typeof (data) != 'object') {
                throw new Error(`Invalid format in deny list file at ${this.bucketName}/${this.objectKey}. Expecting a map`);
            }
            this._map = data;
        }
        catch (e) {
            if (e.code === 'NoSuchKey' || e.code === 'NoSuchBucket') {
                return;
            }
            throw new Error(`Unable to parse deny list file ${this.bucketName}/${this.objectKey}: ${e}`);
        }
    }
    /**
     * Checks if a package (name + version) is in the deny list.
     * @param name The name of the package
     * @param version The package version
     * @returns `undefined` if the package is not in the deny list or a
     * `DenyListRule` otherwise.
     */
    lookup(name, version) {
        var _a;
        if (!this._map) {
            throw new Error('DenyListClient must be initialized');
        }
        return (_a = this._map[name]) !== null && _a !== void 0 ? _a : this._map[`${name}/v${version}`];
    }
    /**
     * Returns a copy of the deny list map.
     */
    get map() {
        if (!this._map) {
            throw new Error('DenyListClient must be initialized');
        }
        return this._map;
    }
}
exports.DenyListClient = DenyListClient;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xpZW50LmxhbWJkYS1zaGFyZWQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvYmFja2VuZC9kZW55LWxpc3QvY2xpZW50LmxhbWJkYS1zaGFyZWQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsbUVBQWlEO0FBQ2pELG1FQUF5RDtBQUV6RCwyQ0FBa0Y7QUFFbEY7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUFnQnpCO1FBQ0UsSUFBSSxDQUFDLFVBQVUsR0FBRyw4QkFBVSxDQUFDLHFDQUF5QixDQUFDLENBQUM7UUFDeEQsSUFBSSxDQUFDLFNBQVMsR0FBRyw4QkFBVSxDQUFDLG9DQUF3QixDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLEVBQUUsR0FBRyxzQkFBRSxFQUFFLENBQUM7SUFDakIsQ0FBQztJQW5CRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsU0FBUztRQUMzQixNQUFNLE1BQU0sR0FBRyxJQUFJLGNBQWMsRUFBRSxDQUFDO1FBQ3BDLE1BQU0sTUFBTSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ3BCLE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFjRDs7OztPQUlHO0lBQ0ssS0FBSyxDQUFDLElBQUk7UUFDaEIsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ2IsTUFBTSxJQUFJLEtBQUssQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1NBQ2xEO1FBRUQsSUFBSSxDQUFDLElBQUksR0FBRyxFQUFFLENBQUM7UUFFZixJQUFJO1lBQ0YsTUFBTSxNQUFNLEdBQUc7Z0JBQ2IsTUFBTSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUN2QixHQUFHLEVBQUUsSUFBSSxDQUFDLFNBQVM7YUFDcEIsQ0FBQztZQUVGLE1BQU0sRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLEdBQUcsTUFBTSxJQUFJLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUNqRSxJQUFJLENBQUMsSUFBSSxFQUFFO2dCQUNULE9BQU8sQ0FBQyxHQUFHLENBQUMsdUNBQXVDLElBQUksQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7Z0JBQ3hGLE9BQU87YUFDUjtZQUVELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7WUFFeEMsK0NBQStDO1lBQy9DLElBQUksUUFBUSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7Z0JBQ3pCLE9BQU87YUFDUjtZQUVELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFnQixDQUFDO1lBQ2pELElBQUksT0FBTSxDQUFDLElBQUksQ0FBQyxJQUFJLFFBQVEsRUFBRTtnQkFDNUIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1Q0FBdUMsSUFBSSxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsU0FBUyxtQkFBbUIsQ0FBQyxDQUFDO2FBQzlHO1lBRUQsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUM7U0FDbEI7UUFBQyxPQUFPLENBQUMsRUFBRTtZQUNWLElBQUksQ0FBQyxDQUFDLElBQUksS0FBSyxXQUFXLElBQUksQ0FBQyxDQUFDLElBQUksS0FBSyxjQUFjLEVBQUU7Z0JBQ3ZELE9BQU87YUFDUjtZQUVELE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLElBQUksQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLFNBQVMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQzlGO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxJQUFZLEVBQUUsT0FBZTs7UUFDekMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7U0FDdkQ7UUFFRCxhQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLG1DQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLEtBQUssT0FBTyxFQUFFLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLEdBQUc7UUFDWixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRTtZQUNkLE1BQU0sSUFBSSxLQUFLLENBQUMsb0NBQW9DLENBQUMsQ0FBQztTQUN2RDtRQUVELE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQztJQUNuQixDQUFDO0NBQ0Y7QUE3RkQsd0NBNkZDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgQVdTIGZyb20gJ2F3cy1zZGsnO1xuaW1wb3J0IHsgczMgfSBmcm9tICcuLi9zaGFyZWQvYXdzLmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0IHsgcmVxdWlyZUVudiB9IGZyb20gJy4uL3NoYXJlZC9lbnYubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBEZW55TGlzdE1hcCwgRGVueUxpc3RSdWxlIH0gZnJvbSAnLi9hcGknO1xuaW1wb3J0IHsgRU5WX0RFTllfTElTVF9CVUNLRVRfTkFNRSwgRU5WX0RFTllfTElTVF9PQkpFQ1RfS0VZIH0gZnJvbSAnLi9jb25zdGFudHMnO1xuXG4vKipcbiAqIEEgY2xpZW50IGZvciB3b3JraW5nIHdpdGggdGhlIGRlbnkgbGlzdC5cbiAqL1xuZXhwb3J0IGNsYXNzIERlbnlMaXN0Q2xpZW50IHtcbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgY2xpZW50IGZvciBhY2Nlc3NpbmcgdGhlIGRlbnkgbGlzdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYXN5bmMgbmV3Q2xpZW50KCkge1xuICAgIGNvbnN0IGNsaWVudCA9IG5ldyBEZW55TGlzdENsaWVudCgpO1xuICAgIGF3YWl0IGNsaWVudC5pbml0KCk7XG4gICAgcmV0dXJuIGNsaWVudDtcbiAgfVxuXG4gIHByaXZhdGUgcmVhZG9ubHkgczM6IEFXUy5TMztcbiAgcHJpdmF0ZSByZWFkb25seSBidWNrZXROYW1lOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgb2JqZWN0S2V5OiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBfbWFwOiBEZW55TGlzdE1hcCB8IHVuZGVmaW5lZDtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKCkge1xuICAgIHRoaXMuYnVja2V0TmFtZSA9IHJlcXVpcmVFbnYoRU5WX0RFTllfTElTVF9CVUNLRVRfTkFNRSk7XG4gICAgdGhpcy5vYmplY3RLZXkgPSByZXF1aXJlRW52KEVOVl9ERU5ZX0xJU1RfT0JKRUNUX0tFWSk7XG4gICAgdGhpcy5zMyA9IHMzKCk7XG4gIH1cblxuICAvKipcbiAgICogRG93bmxvYWRzIHRoZSBkZW55IGxpc3QgYW5kIHN0b3JlcyBpdCBpbiBtZW1vcnkuXG4gICAqXG4gICAqIFRoaXMgbXVzdCBiZSBjYWxsZWQgYmVmb3JlIGBsb29rdXAoKWAuXG4gICAqL1xuICBwcml2YXRlIGFzeW5jIGluaXQoKSB7XG4gICAgaWYgKHRoaXMuX21hcCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdpbml0KCkgY2Fubm90IGJlIGNhbGxlZCB0d2ljZScpO1xuICAgIH1cblxuICAgIHRoaXMuX21hcCA9IHt9O1xuXG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHBhcmFtcyA9IHtcbiAgICAgICAgQnVja2V0OiB0aGlzLmJ1Y2tldE5hbWUsXG4gICAgICAgIEtleTogdGhpcy5vYmplY3RLZXksXG4gICAgICB9O1xuXG4gICAgICBjb25zdCB7IEJvZHk6IGJvZHkgfSA9IGF3YWl0IHRoaXMuczMuZ2V0T2JqZWN0KHBhcmFtcykucHJvbWlzZSgpO1xuICAgICAgaWYgKCFib2R5KSB7XG4gICAgICAgIGNvbnNvbGUubG9nKGBXQVJOSU5HOiBkZW55IGxpc3QgYm9keSBpcyBlbXB0eSBhdCAke3RoaXMuYnVja2V0TmFtZX0vJHt0aGlzLm9iamVjdEtleX1gKTtcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBjb250ZW50cyA9IGJvZHkudG9TdHJpbmcoJ3V0Zi04Jyk7XG5cbiAgICAgIC8vIGFuIGVtcHR5IHN0cmluZyBpcyBhIHZhbGlkIChlbXB0eSkgZGVueSBsaXN0XG4gICAgICBpZiAoY29udGVudHMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cblxuICAgICAgY29uc3QgZGF0YSA9IEpTT04ucGFyc2UoY29udGVudHMpIGFzIERlbnlMaXN0TWFwO1xuICAgICAgaWYgKHR5cGVvZihkYXRhKSAhPSAnb2JqZWN0Jykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgZm9ybWF0IGluIGRlbnkgbGlzdCBmaWxlIGF0ICR7dGhpcy5idWNrZXROYW1lfS8ke3RoaXMub2JqZWN0S2V5fS4gRXhwZWN0aW5nIGEgbWFwYCk7XG4gICAgICB9XG5cbiAgICAgIHRoaXMuX21hcCA9IGRhdGE7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgaWYgKGUuY29kZSA9PT0gJ05vU3VjaEtleScgfHwgZS5jb2RlID09PSAnTm9TdWNoQnVja2V0Jykge1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG5cbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIHBhcnNlIGRlbnkgbGlzdCBmaWxlICR7dGhpcy5idWNrZXROYW1lfS8ke3RoaXMub2JqZWN0S2V5fTogJHtlfWApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVja3MgaWYgYSBwYWNrYWdlIChuYW1lICsgdmVyc2lvbikgaXMgaW4gdGhlIGRlbnkgbGlzdC5cbiAgICogQHBhcmFtIG5hbWUgVGhlIG5hbWUgb2YgdGhlIHBhY2thZ2VcbiAgICogQHBhcmFtIHZlcnNpb24gVGhlIHBhY2thZ2UgdmVyc2lvblxuICAgKiBAcmV0dXJucyBgdW5kZWZpbmVkYCBpZiB0aGUgcGFja2FnZSBpcyBub3QgaW4gdGhlIGRlbnkgbGlzdCBvciBhXG4gICAqIGBEZW55TGlzdFJ1bGVgIG90aGVyd2lzZS5cbiAgICovXG4gIHB1YmxpYyBsb29rdXAobmFtZTogc3RyaW5nLCB2ZXJzaW9uOiBzdHJpbmcpOiBEZW55TGlzdFJ1bGUgfCB1bmRlZmluZWQge1xuICAgIGlmICghdGhpcy5fbWFwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0RlbnlMaXN0Q2xpZW50IG11c3QgYmUgaW5pdGlhbGl6ZWQnKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5fbWFwW25hbWVdID8/IHRoaXMuX21hcFtgJHtuYW1lfS92JHt2ZXJzaW9ufWBdO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBjb3B5IG9mIHRoZSBkZW55IGxpc3QgbWFwLlxuICAgKi9cbiAgcHVibGljIGdldCBtYXAoKSB7XG4gICAgaWYgKCF0aGlzLl9tYXApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRGVueUxpc3RDbGllbnQgbXVzdCBiZSBpbml0aWFsaXplZCcpO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLl9tYXA7XG4gIH1cbn1cbiJdfQ==