"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const CONFIG_BUCKET_NAME = env_lambda_shared_1.requireEnv('CONFIG_BUCKET_NAME');
    const CONFIG_FILE_KEY = env_lambda_shared_1.requireEnv('CONFIG_FILE_KEY');
    // Load configuration
    const { packageTags: packageTagsConfig, packageLinks: allowedLinks } = await getConfig(CONFIG_BUCKET_NAME, CONFIG_FILE_KEY);
    const codeArtifactProps = (function () {
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            return undefined;
        }
        const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
        const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
        const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
        return { endpoint, domain, domainOwner, apiEndpoint };
    })();
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const { integrity: integrityCheck } = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let parsedAssembly;
        let constructFramework;
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            parsedAssembly = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            // needs `dependencyClosure`
            constructFramework = detectConstructFramework(parsedAssembly);
            const { license, name, version } = parsedAssembly;
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            // Delete some fields not used by the client to reduce the size of the assembly.
            // See https://github.com/cdklabs/construct-hub-webapp/issues/691
            delete parsedAssembly.types;
            delete parsedAssembly.readme;
            delete parsedAssembly.dependencyClosure;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, packageJsonObj)) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        if (codeArtifactProps) {
            console.log('Publishing to the internal CodeArtifact...');
            try {
                const { publishConfig } = packageJsonObj;
                if (publishConfig) {
                    console.log('Not publishing to CodeArtifact due to the presence of publishConfig in package.json: ', publishConfig);
                }
                else {
                    await code_artifact_lambda_shared_1.codeArtifactPublishPackage(Buffer.from(tarball.Body), codeArtifactProps);
                }
            }
            catch (err) {
                console.error('Failed publishing to CodeArtifact: ', err);
            }
        }
        const metadata = {
            constructFramework,
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public, max-age=86400, must-revalidate, s-maxage=300, proxy-revalidate',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: Buffer.from(JSON.stringify(parsedAssembly), 'utf-8'),
            CacheControl: 'public, max-age: 86400, must-revalidate, s-maxage=300, proxy-revalidate',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Determines the Construct framework used by the provided assembly.
 *
 * @param assembly the assembly for which a construct framework should be
 *                 identified.
 *
 * @returns a construct framework if one could be identified.
 */
function detectConstructFramework(assembly) {
    var _a;
    let name;
    let nameAmbiguous = false;
    let majorVersion;
    let majorVersionAmbiguous = false;
    detectConstructFrameworkPackage(assembly.name, assembly.version);
    for (const depName of Object.keys((_a = assembly.dependencyClosure) !== null && _a !== void 0 ? _a : {})) {
        detectConstructFrameworkPackage(depName);
        if (nameAmbiguous) {
            return undefined;
        }
    }
    return name && { name, majorVersion: majorVersionAmbiguous ? undefined : majorVersion };
    function detectConstructFrameworkPackage(packageName, versionRange) {
        var _a, _b;
        if (versionRange === void 0) { versionRange = (_a = assembly.dependencies) === null || _a === void 0 ? void 0 : _a[packageName]; }
        if (packageName.startsWith('@aws-cdk/') || packageName === 'aws-cdk-lib' || packageName === 'monocdk') {
            if (name && name !== "aws-cdk" /* AWS_CDK */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "aws-cdk" /* AWS_CDK */;
        }
        else if (packageName === 'cdktf' || packageName.startsWith('@cdktf/')) {
            if (name && name !== "cdktf" /* CDKTF */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdktf" /* CDKTF */;
        }
        else if (packageName === 'cdk8s' || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(packageName)) {
            if (name && name !== "cdk8s" /* CDK8S */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdk8s" /* CDK8S */;
        }
        else {
            return;
        }
        if (versionRange) {
            const major = (_b = semver_1.minVersion(versionRange)) === null || _b === void 0 ? void 0 : _b.major;
            if (majorVersion != null && majorVersion !== major) {
                // Identified multiple candidates, so this is ambiguous...
                majorVersionAmbiguous = true;
            }
            majorVersion = major;
        }
        return;
    }
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
/**
 * Looks for the ingestion configuration file in the passed bucket and parses
 * it. If it is not found or invalid then a default is returned.
 */
async function getConfig(bucket, key) {
    var _a;
    const defaultConfig = {
        packageTags: [],
        packageLinks: [],
    };
    try {
        const req = await aws.s3().getObject({
            Bucket: bucket,
            Key: key,
        }).promise();
        const body = (_a = req === null || req === void 0 ? void 0 : req.Body) === null || _a === void 0 ? void 0 : _a.toString();
        if (body) {
            return JSON.parse(body);
        }
        return defaultConfig;
    }
    catch (e) {
        console.error(e);
        return defaultConfig;
    }
}
//# sourceMappingURL=data:application/json;base64,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