"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const acm = require("@aws-cdk/aws-certificatemanager");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const route53 = require("@aws-cdk/aws-route53");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * Elasticsearch version.
 *
 * @stability stable
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom Elasticsearch version.
     *
     * @param version custom version number.
     * @stability stable
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
_a = JSII_RTTI_SYMBOL_1;
ElasticsearchVersion[_a] = { fqn: "@aws-cdk/aws-elasticsearch.ElasticsearchVersion", version: "1.133.0" };
/**
 * AWS Elasticsearch 1.5.
 *
 * @stability stable
 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/**
 * AWS Elasticsearch 2.3.
 *
 * @stability stable
 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/**
 * AWS Elasticsearch 5.1.
 *
 * @stability stable
 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/**
 * AWS Elasticsearch 5.3.
 *
 * @stability stable
 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/**
 * AWS Elasticsearch 5.5.
 *
 * @stability stable
 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/**
 * AWS Elasticsearch 5.6.
 *
 * @stability stable
 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/**
 * AWS Elasticsearch 6.0.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/**
 * AWS Elasticsearch 6.2.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/**
 * AWS Elasticsearch 6.3.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/**
 * AWS Elasticsearch 6.4.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/**
 * AWS Elasticsearch 6.5.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/**
 * AWS Elasticsearch 6.7.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/**
 * AWS Elasticsearch 6.8.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/**
 * AWS Elasticsearch 7.1.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/**
 * AWS Elasticsearch 7.4.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/**
 * AWS Elasticsearch 7.7.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * AWS Elasticsearch 7.8.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_8 = ElasticsearchVersion.of('7.8');
/**
 * AWS Elasticsearch 7.9.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_9 = ElasticsearchVersion.of('7.9');
/**
 * AWS Elasticsearch 7.10.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_10 = ElasticsearchVersion.of('7.10');
/**
 * The minimum TLS version required for traffic to the domain.
 *
 * @stability stable
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this Domain.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensionsMap: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @stability stable
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @stability stable
     */
    metricClusterIndexWritesBlocked(props) {
        return this.metric('ClusterIndexWritesBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @stability stable
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Elasticsearch domain.
 *
 * @stability stable
 */
class Domain extends DomainBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _30, _31, _32;
        super(scope, id, {
            physicalName: props.domainName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_elasticsearch_DomainProps(props);
        const defaultInstanceType = 'r5.large.elasticsearch';
        const warmDefaultInstanceType = 'ultrawarm1.medium.elasticsearch';
        const dedicatedMasterType = initializeInstanceType(defaultInstanceType, (_c = props.capacity) === null || _c === void 0 ? void 0 : _c.masterNodeInstanceType);
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = cdk.Token.isUnresolved(dedicatedMasterCount) ? true : dedicatedMasterCount > 0;
        const instanceType = initializeInstanceType(defaultInstanceType, (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType);
        const instanceCount = (_h = (_g = props.capacity) === null || _g === void 0 ? void 0 : _g.dataNodes) !== null && _h !== void 0 ? _h : 1;
        const warmType = initializeInstanceType(warmDefaultInstanceType, (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.warmInstanceType);
        const warmCount = (_l = (_k = props.capacity) === null || _k === void 0 ? void 0 : _k.warmNodes) !== null && _l !== void 0 ? _l : 0;
        const warmEnabled = cdk.Token.isUnresolved(warmCount) ? true : warmCount > 0;
        const availabilityZoneCount = (_o = (_m = props.zoneAwareness) === null || _m === void 0 ? void 0 : _m.availabilityZoneCount) !== null && _o !== void 0 ? _o : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_q = (_p = props.zoneAwareness) === null || _p === void 0 ? void 0 : _p.enabled) !== null && _q !== void 0 ? _q : ((_r = props.zoneAwareness) === null || _r === void 0 ? void 0 : _r.availabilityZoneCount) != null;
        let securityGroups;
        let subnets;
        if (props.vpc) {
            subnets = selectSubnets(props.vpc, (_s = props.vpcSubnets) !== null && _s !== void 0 ? _s : [{ subnetType: ec2.SubnetType.PRIVATE }]);
            securityGroups = (_t = props.securityGroups) !== null && _t !== void 0 ? _t : [new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                    description: `Security group for domain ${this.node.id}`,
                })];
            this._connections = new ec2.Connections({ securityGroups });
        }
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (subnets && zoneAwarenessEnabled && new Set(subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        if ([dedicatedMasterType, instanceType, warmType].some(t => (!cdk.Token.isUnresolved(t) && !t.endsWith('.elasticsearch')))) {
            throw new Error('Master, data and UltraWarm node instance types must end with ".elasticsearch".');
        }
        if (!cdk.Token.isUnresolved(warmType) && !warmType.startsWith('ultrawarm')) {
            throw new Error('UltraWarm node instance type must start with "ultrawarm".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_u = props.useUnsignedBasicAuth) !== null && _u !== void 0 ? _u : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_v = props.encryptionAtRest) === null || _v === void 0 ? void 0 : _v.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.AnyPrincipal()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserArn;
        const masterUserNameProps = (_x = props.fineGrainedAccessControl) === null || _x === void 0 ? void 0 : _x.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_y = props.fineGrainedAccessControl) === null || _y === void 0 ? void 0 : _y.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_0 = (_z = props.encryptionAtRest) === null || _z === void 0 ? void 0 : _z.enabled) !== null && _0 !== void 0 ? _0 : (((_1 = props.encryptionAtRest) === null || _1 === void 0 ? void 0 : _1.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_2 = props.nodeToNodeEncryption) !== null && _2 !== void 0 ? _2 : unsignedBasicAuthEnabled;
        const volumeSize = (_4 = (_3 = props.ebs) === null || _3 === void 0 ? void 0 : _3.volumeSize) !== null && _4 !== void 0 ? _4 : 10;
        const volumeType = (_6 = (_5 = props.ebs) === null || _5 === void 0 ? void 0 : _5.volumeType) !== null && _6 !== void 0 ? _6 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_8 = (_7 = props.ebs) === null || _7 === void 0 ? void 0 : _7.enabled) !== null && _8 !== void 0 ? _8 : true;
        const enforceHttps = (_9 = props.enforceHttps) !== null && _9 !== void 0 ? _9 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if ((_10 = props.logging) === null || _10 === void 0 ? void 0 : _10.appLogEnabled) {
                throw new Error('Error logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_11 = props.encryptionAtRest) === null || _11 === void 0 ? void 0 : _11.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.8 && warmEnabled) {
            throw new Error('UltraWarm requires Elasticsearch 6.8 or later.');
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isSomeInstanceType('i3', 'r6gd') && ebsEnabled) {
            throw new Error('I3 and R6GD instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        if (isSomeInstanceType('t2', 't3') && warmEnabled) {
            throw new Error('T2 and T3 instance types do not support UltraWarm storage.');
        }
        // Only R3, I3 and r6gd support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3', 'r6gd')) {
            throw new Error('EBS volumes are required when using instance types other than r3, i3 or r6gd.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.auditLogEnabled) && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        // Validate UltraWarm requirement for dedicated master nodes, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/ultrawarm.html
        if (warmEnabled && !dedicatedMasterEnabled) {
            throw new Error('Dedicated master node is required when UltraWarm storage is enabled.');
        }
        let cfnVpcOptions;
        if (securityGroups && subnets) {
            cfnVpcOptions = {
                securityGroupIds: securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_13 = props.logging) === null || _13 === void 0 ? void 0 : _13.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_14 = props.logging.slowSearchLogGroup) !== null && _14 !== void 0 ? _14 : new logs.LogGroup(this, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_15 = props.logging) === null || _15 === void 0 ? void 0 : _15.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_16 = props.logging.slowIndexLogGroup) !== null && _16 !== void 0 ? _16 : new logs.LogGroup(this, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_17 = props.logging) === null || _17 === void 0 ? void 0 : _17.appLogEnabled) {
            this.appLogGroup = (_18 = props.logging.appLogGroup) !== null && _18 !== void 0 ? _18 : new logs.LogGroup(this, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        if ((_19 = props.logging) === null || _19 === void 0 ? void 0 : _19.auditLogEnabled) {
            this.auditLogGroup = (_20 = props.logging.auditLogGroup) !== null && _20 !== void 0 ? _20 : new logs.LogGroup(this, 'AuditLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: (_21 = this.auditLogGroup) === null || _21 === void 0 ? void 0 : _21.logGroupArn,
            };
        }
        let customEndpointCertificate;
        if (props.customEndpoint) {
            if (props.customEndpoint.certificate) {
                customEndpointCertificate = props.customEndpoint.certificate;
            }
            else {
                customEndpointCertificate = new acm.Certificate(this, 'CustomEndpointCertificate', {
                    domainName: props.customEndpoint.domainName,
                    validation: props.customEndpoint.hostedZone ? acm.CertificateValidation.fromDns(props.customEndpoint.hostedZone) : undefined,
                });
            }
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                warmEnabled: warmEnabled
                    ? warmEnabled
                    : undefined,
                warmCount: warmEnabled
                    ? warmCount
                    : undefined,
                warmType: warmEnabled
                    ? warmType
                    : undefined,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_22 = props.ebs) === null || _22 === void 0 ? void 0 : _22.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_24 = (_23 = props.encryptionAtRest) === null || _23 === void 0 ? void 0 : _23.kmsKey) === null || _24 === void 0 ? void 0 : _24.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_25 = props.cognitoKibanaAuth) === null || _25 === void 0 ? void 0 : _25.identityPoolId,
                roleArn: (_26 = props.cognitoKibanaAuth) === null || _26 === void 0 ? void 0 : _26.role.roleArn,
                userPoolId: (_27 = props.cognitoKibanaAuth) === null || _27 === void 0 ? void 0 : _27.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_28 = props.tlsSecurityPolicy) !== null && _28 !== void 0 ? _28 : TLSSecurityPolicy.TLS_1_0,
                ...props.customEndpoint && {
                    customEndpointEnabled: true,
                    customEndpoint: props.customEndpoint.domainName,
                    customEndpointCertificateArn: customEndpointCertificate.certificateArn,
                },
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_29 = this.masterUserPassword) === null || _29 === void 0 ? void 0 : _29.toString(),
                    },
                }
                : undefined,
            advancedOptions: props.advancedOptions,
        });
        this.domain.applyRemovalPolicy(props.removalPolicy);
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            if (!cdk.Token.isUnresolved(props.domainName)) {
                // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/configuration-api.html#configuration-api-datatypes-domainname
                if (!props.domainName.match(/^[a-z0-9\-]+$/)) {
                    throw new Error(`Invalid domainName '${props.domainName}'. Valid characters are a-z (lowercase only), 0-9, and – (hyphen).`);
                }
                if (props.domainName.length < 3 || props.domainName.length > 28) {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must be between 3 and 28 characters`);
                }
                if (props.domainName[0] < 'a' || props.domainName[0] > 'z') {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must start with a lowercase letter`);
                }
            }
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        if ((_30 = props.customEndpoint) === null || _30 === void 0 ? void 0 : _30.hostedZone) {
            new route53.CnameRecord(this, 'CnameRecord', {
                recordName: props.customEndpoint.domainName,
                zone: props.customEndpoint.hostedZone,
                domainName: this.domainEndpoint,
            });
        }
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_31 = props.accessPolicies) !== null && _31 !== void 0 ? _31 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            if ((_32 = props.encryptionAtRest) === null || _32 === void 0 ? void 0 : _32.kmsKey) {
                // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/encryption-at-rest.html
                // these permissions are documented as required during domain creation.
                // while not strictly documented for updates as well, it stands to reason that an update
                // operation might require these in case the cluster uses a kms key.
                // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                    actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                    resources: [props.encryptionAtRest.kmsKey.keyArn],
                    effect: iam.Effect.ALLOW,
                }));
            }
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @stability stable
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @stability stable
     */
    static fromDomainAttributes(scope, id, attrs) {
        var _c;
        jsiiDeprecationWarnings._aws_cdk_aws_elasticsearch_DomainAttributes(attrs);
        const { domainArn, domainEndpoint } = attrs;
        const domainName = (_c = cdk.Stack.of(scope).splitArn(domainArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName) !== null && _c !== void 0 ? _c : extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
    /**
     * Manages network connections to the domain.
     *
     * This will throw an error in case the domain
     * is not placed inside a VPC.
     *
     * @stability stable
     */
    get connections() {
        if (!this._connections) {
            throw new Error("Connections are only available on VPC enabled domains. Use the 'vpc' property to place a domain inside a VPC");
        }
        return this._connections;
    }
}
exports.Domain = Domain;
_b = JSII_RTTI_SYMBOL_1;
Domain[_b] = { fqn: "@aws-cdk/aws-elasticsearch.Domain", version: "1.133.0" };
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
function selectSubnets(vpc, vpcSubnets) {
    const selected = [];
    for (const selection of vpcSubnets) {
        selected.push(...vpc.selectSubnets(selection).subnets);
    }
    return selected;
}
/**
 * Initializes an instance type.
 *
 * @param defaultInstanceType Default instance type which is used if no instance type is provided
 * @param instanceType Instance type
 * @returns Instance type in lowercase (if provided) or default instance type
 */
function initializeInstanceType(defaultInstanceType, instanceType) {
    if (instanceType) {
        return cdk.Token.isUnresolved(instanceType) ? instanceType : instanceType.toLowerCase();
    }
    else {
        return defaultInstanceType;
    }
}
//# sourceMappingURL=data:application/json;base64,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